import contextlib
import enum
import hashlib
import inspect
import os
import re
import shutil
import subprocess
import time
import uuid
from collections import deque
from numbers import Number
from pathlib import Path
from queue import Queue
from threading import Thread
from typing import (
    Any,
    Callable,
    Dict,
    Generator,
    Iterable,
    List,
    Optional,
    Tuple,
    Union,
)

import click
from attrs import Factory, define
from loguru import logger as log

# Fixme: Part of me wants to patch __builtins__, but it's bad for autocompletion and clarity

#ffmpeg =  "./bin/ff/ffmpeg.exe"
#ffprobe = "./bin/ff/ffprobe.exe"

def flatten(
    *items: Union[Any, Iterable[Any]],
    cast: Optional[type]=list,
    block: Optional[List[Any]]=[None, ""],
    unpack: Tuple[type]=(list, deque, tuple, map, Generator),
) -> List[Any]:
    """
    Flatten/unpack nested iterables (list, deque, tuple, map, Generator) to a plain 1D list
    - Removes common falsy values by default, disable with `block={None, False, "", [], ...}`

    Usage:
        ```python
        flatten([1, [2, 3], 4, [5, [6, 7]]]) # [1, 2, 3, 4, 5, 6, 7]
        flatten(range(3), (True, False), None, "Hello") # [0, 1, 2, True, False, "Hello"]
        ```
    """
    def flatten(stuff):
        if bool(block):
            stuff = filter(lambda item: (item not in block), stuff)

        for item in stuff:
            if isinstance(item, unpack):
                yield from flatten(item)
                continue
            yield item

    # Note: Recursive implementation
    return cast(flatten(items))

def every(
    *items: Union[Any, Iterable[Any]],
    cast: Optional[type]=list,
    block: List[Any]=[None, ""],
) -> Optional[List[Any]]:
    """
    Returns the flattened items if not any element is in the block list, else None. Useful when
    a Model class has a list of optional arguments that doesn't add falsy values to a command

    Usage:
        ```python
        valid(1, 2, 3) # [1, 2, 3]
        valid(1, 2, 3, None) # None
        valid("value", value, block=[0, 2]) # None if value is 0 or 2, else ["value", value]
        ```
    """
    items = flatten(*items, block=None, cast=cast)
    if any(item in block for item in items):
        return None
    return items

def shell(
    *args: Iterable[Any],
    output: bool=False,
    Popen: bool=False,
    env: Dict[str, str]=None,
    confirm: bool=False,
    threaded_stdin: bool=False,
    skip: bool=False,
    echo: bool=True,
    **kwargs
) -> Union[None, str, subprocess.Popen, subprocess.CompletedProcess]:
    """
    Enhanced subprocess runners with many additional features. Flattens the args, converts to str

    Example:
        ```python
        shell(["binary", "-m"], "arg1", None, "arg2", 3, confirm=True)
        ```
    """
    if (output and Popen):
        raise ValueError(log.error("Cannot use (output=True) and (Popen=True) at the same time"))

    args = tuple(map(str, flatten(args)))

    # Assert command won't fail due unknown binary
    if (not shutil.which(args[0])):
        raise FileNotFoundError(log.error(f"Binary doesn't exist or was not found on PATH ({args[0]})"))

    # Log the command being run, temp variables
    _cwd = f" @ ({kwargs.get('cwd', '') or Path.cwd()})"
    _log = (log.skip if skip else log.info)
    _the = ("Skipping" if skip else "Running")
    _log(_the + f" Command {args}{_cwd}", echo=echo)
    if skip: return

    if kwargs.get("shell", False):
        args = '"' + '" "'.join(args) + '"'
        log.warning((
            "Running command with (shell=True), be careful.. "
            "Consider using (confirm=True)"*(not confirm)
        ))

    if confirm and not click.confirm("• Confirm running the command above"):
        return

    # Update current environ for the command only
    kwargs["env"] = os.environ | (env or {})

    # Windows: preexec_fn is not supported, remove from kwargs
    if (os.name == "nt") and (kwargs.pop("preexec_fn", None)):
        log.minor("shell(preexec_fn=...) is not supported on Windows, ignoring..")

    if output:
        return subprocess.check_output(args, **kwargs).decode("utf-8")

    elif Popen:
        process = subprocess.Popen(args, **kwargs)

        if bool(threaded_stdin):

            @define
            class StdinWrapper:
                _process: subprocess.Popen
                _queue: Queue = Factory(factory=lambda: Queue(maxsize=10))
                _loop: bool = True
                _stdin: Any = None

                def __attrs_post_init__(self):
                    Thread(target=self.worker, daemon=True).start()
                def write(self, data):
                    self._queue.put(data)
                def worker(self):
                    while self._loop:
                        self._stdin.write(self._queue.get())
                        self._queue.task_done()
                def close(self):
                    self._queue.join()
                    self._stdin.close()
                    self._loop = False
                    while self._process.poll() is None:
                        time.sleep(0.01)

            process.stdin = StdinWrapper(process=process, stdin=process.stdin)
        return process
    else:
        return subprocess.run(args, **kwargs)

def clamp(value: float, low: float=0, high: float=1) -> float:
    return max(low, min(value, high))

def apply(callback: Callable, iterable: Iterable[Any], *, cast: Callable=list) -> List[Any]:
    """Applies a callback to all items of an iterable, returning a $cast of the results"""
    return cast(map(callback, iterable))

def denum(item: Union[enum.Enum, Any]) -> Any:
    """De-enumerates an item, if it's an Enum returns the value, else the item itself"""
    return (item.value if isinstance(item, enum.Enum) else item)

def pydantic_cli(instance: object, post: Callable=None) -> Callable:
    """Makes a Pydantic BaseModel class signature Typer compatible, by copying the class's signature
    to a wrapper virtual function. All kwargs sent are set as attributes on the instance, and typer
    will send all default ones overriden by the user commands. The 'post' method is called afterwards,
    for example `post = self.set_object` for back-communication between the caller and the instance"""
    from pydantic import BaseModel

    if not issubclass(this := type(instance), BaseModel):
        raise TypeError(f"Object {this} is not a Pydantic BaseModel")

    def wrapper(**kwargs):
        for name, value in kwargs.items():
            setattr(instance, name, value)
        if post: post(instance)
    wrapper.__signature__ = inspect.signature(instance.__class__)
    wrapper.__doc__ = instance.__doc__
    return wrapper

def nearest(number: Number, multiple: Number, *, type=int, operator: Callable=round) -> Number:
    """Finds the nearest multiple of a base number, by default ints but works for floats too"""
    return type(multiple * operator(number/multiple))

def dunder(name: str) -> bool:
    return name.startswith("__") and name.endswith("__")

def hyphen_range(string: Optional[str], *, inclusive: bool=True) -> Generator[int, None, None]:
    """
    Yields the numbers in a hyphenated CSV range, just like when selecting what pages to print
    - Accepts any of ("-", "..", "...", "_", "->") as a hyphenated range

    Example:
        ```python
        hyphen_range("2,3") # 2, 3
        hyphen_range("2-5") # 2, 3, 4, 5
        hyphen_range("1-3, 5") # 1, 2, 3, 5
        ```
    """
    if not bool(string):
        return None

    for part in string.split(","):
        if ("-" in part):
            start, end = map(int, re.split(r"_|-|\.\.|\.\.\.|\-\>", part))
            yield from range(start, end + int(inclusive))
            continue
        yield int(part)

def image_hash(image) -> str:
    """A Fast-ish method to get an object's hash that implements .tobytes()"""
    return str(uuid.UUID(hashlib.sha256(image.tobytes()).hexdigest()[::2]))

def limited_integer_ratio(number: Optional[float], *, limit: float=None) -> Optional[Tuple[int, int]]:
    """Same as Number.as_integer_ratio but with an optional upper limit and optional return"""
    if number is None:
        return None

    num, den = number.as_integer_ratio()

    if limit and (den > limit or num > limit):
        normalize = limit/min(num, den)
        num, den = int(num * normalize), int(den * normalize)

    return num, den

@contextlib.contextmanager
def temp_env(**env: Dict[str, str]) -> Generator[None, None, None]:
    """Temporarily sets environment variables inside a context"""
    old = os.environ.copy()
    os.environ.clear()
    os.environ.update({k: str(v) for k, v in (old | env).items() if v})
    yield
    os.environ.clear()
    os.environ.update(old)

@contextlib.contextmanager
def Stack(*contexts: contextlib.AbstractContextManager) -> Generator[None, None, None]:
    """Enter multiple contexts at once as `with Stack(open() as f1, open() as f2): ...`"""
    with contextlib.ExitStack() as stack:
        for context in flatten(contexts):
            stack.enter_context(context)
        yield

def filter_dict(
    data: Dict,
    *,
    block: Optional[Iterable[Any]]=None,
    allow: Optional[Iterable[Any]]=None,
):
    if block:
        data = {key: value for key, value in data.items() if key not in block}
    if allow:
        data = {key: value for key, value in data.items() if key in allow}
    return data

def selfless(data: Dict) -> Dict:
    return filter_dict(data, block=["self"])

# -------------------------------------------------------------------------------------------------|

def transcends(method, base, generator: bool=False):
    """
    Are you tired of managing and calling super().<name>(*args, **kwargs) in your methods?
    > We have just the right solution for you!

    Introducing transcends, the decorator that crosses your class's MRO and calls the method
    with the same name as the one you are decorating. It's an automatic super() !
    """
    name = method.__name__

    def decorator(func: Callable) -> Callable:
        def get_targets(self):
            for cls in type(self).mro()[:-2]:
                if cls in (base, object):
                    continue
                if (target := cls.__dict__.get(name)):
                    yield target

        # Note: We can't have a `if generator` else the func becomes a Generator
        def yields(self, *args, **kwargs):
            for target in get_targets(self):
                yield from target(self, *args, **kwargs)
        def plain(self, *args, **kwargs):
            for target in get_targets(self):
                target(self, *args, **kwargs)

        return (yields if generator else plain)
    return decorator
